/*
 * Decompiled with CFR 0.152.
 */
package org.apache.shindig.gadgets.features;

import com.google.common.collect.Lists;
import com.google.common.collect.MapMaker;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.google.inject.name.Named;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.lang.StringUtils;
import org.apache.shindig.common.Pair;
import org.apache.shindig.common.uri.Uri;
import org.apache.shindig.common.uri.UriBuilder;
import org.apache.shindig.common.util.ResourceLoader;
import org.apache.shindig.gadgets.GadgetContext;
import org.apache.shindig.gadgets.GadgetException;
import org.apache.shindig.gadgets.RenderingContext;
import org.apache.shindig.gadgets.features.FeatureParser;
import org.apache.shindig.gadgets.features.FeatureResource;
import org.apache.shindig.gadgets.features.FeatureResourceLoader;

@Singleton
public class FeatureRegistry {
    public static final char FILE_SEPARATOR = ',';
    public static final String RESOURCE_SCHEME = "res";
    public static final String FILE_SCHEME = "file";
    private static final Logger logger = Logger.getLogger("org.apache.shindig.gadgets");
    private final Map<Collection<String>, List<FeatureResource>> cache = new MapMaker().makeMap();
    private final Map<Collection<String>, List<FeatureResource>> cacheIgnoreUnsupported = new MapMaker().makeMap();
    private final FeatureParser parser = new FeatureParser();
    private final FeatureResourceLoader resourceLoader;
    private final Map<String, FeatureNode> featureMap;

    @Inject
    public FeatureRegistry(FeatureResourceLoader resourceLoader) {
        this.resourceLoader = resourceLoader;
        this.featureMap = Maps.newHashMap();
    }

    @Inject(optional=true)
    public void addDefaultFeatures(@Named(value="shindig.features.default") String featureFiles) throws GadgetException {
        this.register(featureFiles);
    }

    public void register(String resourceKey) throws GadgetException {
        try {
            for (String location : StringUtils.split((String)resourceKey, (char)',')) {
                Uri uriLoc = FeatureRegistry.getComponentUri(location);
                if (uriLoc.getScheme() != null && uriLoc.getScheme().equals(RESOURCE_SCHEME)) {
                    ArrayList resources = Lists.newArrayList();
                    location = uriLoc.getPath();
                    if (location.startsWith("/")) {
                        location = location.substring(1);
                    }
                    logger.info("Loading resources from: " + uriLoc.toString());
                    if (location.endsWith(".txt")) {
                        for (String resource : this.getResourceContent(location).split("[\r\n]+")) {
                            if ((resource = resource.trim()).length() <= 0 || resource.charAt(0) == '#') continue;
                            resource = FeatureRegistry.getComponentUri(resource.trim()).getPath();
                            resources.add(resource);
                        }
                    } else {
                        resources.add(location);
                    }
                    this.loadResources(resources);
                    continue;
                }
                logger.info("Loading files from: " + location);
                this.loadFile(new File(uriLoc.getPath()));
            }
            this.connectDependencyGraph();
        }
        catch (IOException e) {
            throw new GadgetException(GadgetException.Code.INVALID_PATH, (Throwable)e);
        }
    }

    public List<FeatureResource> getFeatureResources(GadgetContext ctx, Collection<String> needed, List<String> unsupported, boolean transitive) {
        Map<Collection<String>, List<FeatureResource>> useCache = null;
        if (transitive) {
            useCache = unsupported != null ? this.cache : this.cacheIgnoreUnsupported;
        }
        LinkedList resources = Lists.newLinkedList();
        if (useCache != null && useCache.containsKey(needed)) {
            return useCache.get(needed);
        }
        List<FeatureNode> featureNodes = null;
        featureNodes = transitive ? this.getTransitiveDeps(needed, unsupported) : this.getRequestedNodes(needed, unsupported);
        String targetBundleType = ctx.getRenderingContext() == RenderingContext.CONTAINER ? "container" : "gadget";
        for (FeatureNode entry : featureNodes) {
            for (FeatureBundle bundle : entry.getBundles()) {
                if (!bundle.getType().equals(targetBundleType) || !this.containerMatch(bundle.getAttribs().get("container"), ctx.getContainer())) continue;
                resources.addAll(bundle.getResources());
            }
        }
        if (useCache != null && (unsupported == null || unsupported.isEmpty())) {
            useCache.put(needed, resources);
        }
        return resources;
    }

    public List<FeatureResource> getFeatureResources(GadgetContext ctx, Collection<String> needed, List<String> unsupported) {
        return this.getFeatureResources(ctx, needed, unsupported, true);
    }

    public List<FeatureResource> getAllFeatures() {
        return this.getFeatureResources(new GadgetContext(), Lists.newArrayList(this.featureMap.keySet()), null);
    }

    public List<String> getFeatures(List<String> needed) {
        List<FeatureNode> fullTree = this.getTransitiveDeps(needed, Lists.newLinkedList());
        LinkedList allFeatures = Lists.newLinkedList();
        for (FeatureNode node : fullTree) {
            allFeatures.add(node.name);
        }
        return allFeatures;
    }

    public List<String> getAllFeatureNames() {
        return Lists.newArrayList(this.featureMap.keySet());
    }

    String getResourceContent(String resource) throws IOException {
        return ResourceLoader.getContent((String)resource);
    }

    static Uri getComponentUri(String str) {
        Uri uri = null;
        uri = str.startsWith("res://") ? new UriBuilder().setScheme(RESOURCE_SCHEME).setPath(str.substring(6)).toUri() : Uri.parse((String)str);
        return uri;
    }

    private List<FeatureNode> getTransitiveDeps(Collection<String> needed, List<String> unsupported) {
        final List<FeatureNode> requested = this.getRequestedNodes(needed, unsupported);
        Comparator<FeatureNode> nodeDepthComparator = new Comparator<FeatureNode>(){

            @Override
            public int compare(FeatureNode one, FeatureNode two) {
                if (one.nodeDepth > two.nodeDepth || one.nodeDepth == two.nodeDepth && requested.indexOf(one) < requested.indexOf(two)) {
                    return -1;
                }
                return 1;
            }
        };
        Collections.sort(requested, nodeDepthComparator);
        HashSet alreadySeen = Sets.newHashSet();
        LinkedList fullDeps = Lists.newLinkedList();
        for (FeatureNode requestedFeature : requested) {
            for (FeatureNode toAdd : requestedFeature.getTransitiveDeps()) {
                if (alreadySeen.contains(toAdd.name)) continue;
                alreadySeen.add(toAdd.name);
                fullDeps.add(toAdd);
            }
        }
        return fullDeps;
    }

    private List<FeatureNode> getRequestedNodes(Collection<String> needed, List<String> unsupported) {
        ArrayList requested = Lists.newArrayList();
        for (String featureName : needed) {
            if (this.featureMap.containsKey(featureName)) {
                requested.add(this.featureMap.get(featureName));
                continue;
            }
            if (unsupported == null) continue;
            unsupported.add(featureName);
        }
        return requested;
    }

    private boolean containerMatch(String containerAttrib, String container) {
        if (containerAttrib == null || containerAttrib.length() == 0) {
            return true;
        }
        HashSet containers = Sets.newHashSet();
        for (String attr : containerAttrib.split(",")) {
            containers.add(attr.trim());
        }
        return containers.contains(container);
    }

    private void connectDependencyGraph() throws GadgetException {
        LinkedList problems = Lists.newLinkedList();
        LinkedList theFeatures = Lists.newLinkedList();
        for (Map.Entry<String, FeatureNode> featureEntry : this.featureMap.entrySet()) {
            String name = featureEntry.getKey();
            FeatureNode feature = featureEntry.getValue();
            for (String rawDep : feature.getRawDeps()) {
                if (!this.featureMap.containsKey(rawDep)) {
                    problems.add("Feature [" + name + "] has dependency on unknown feature: " + rawDep);
                    continue;
                }
                feature.addDep(this.featureMap.get(rawDep));
                theFeatures.add(feature);
            }
        }
        for (FeatureNode feature : theFeatures) {
            try {
                feature.completeNodeGraph();
            }
            catch (GadgetException e) {
                problems.add(e.getMessage());
            }
        }
        if (problems.size() > 0) {
            StringBuilder sb = new StringBuilder();
            sb.append("Problems found processing features:\n");
            for (String problem : problems) {
                sb.append(problem).append("\n");
            }
            throw new GadgetException(GadgetException.Code.INVALID_CONFIG, sb.toString());
        }
    }

    private void loadResources(List<String> resources) throws GadgetException {
        try {
            for (String resource : resources) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.fine("Processing resource: " + resource);
                }
                String content = this.getResourceContent(resource);
                Uri parent = new UriBuilder().setScheme(RESOURCE_SCHEME).setPath(resource).toUri();
                this.loadFeature(parent, content);
            }
        }
        catch (IOException e) {
            throw new GadgetException(GadgetException.Code.INVALID_PATH, (Throwable)e);
        }
    }

    private void loadFile(File file) throws GadgetException, IOException {
        if (!file.exists() || !file.canRead()) {
            throw new GadgetException(GadgetException.Code.INVALID_CONFIG, "Feature file '" + file.getPath() + "' doesn't exist or can't be read");
        }
        File[] toLoad = null;
        toLoad = file.isDirectory() ? file.listFiles() : new File[]{file};
        for (File featureFile : toLoad) {
            if (featureFile.isDirectory()) {
                this.loadFile(featureFile);
                continue;
            }
            if (featureFile.getName().toLowerCase(Locale.ENGLISH).endsWith(".xml")) {
                String content = ResourceLoader.getContent((File)featureFile);
                Uri parent = Uri.fromJavaUri((URI)featureFile.toURI());
                this.loadFeature(parent, content);
                continue;
            }
            if (!logger.isLoggable(Level.FINEST)) continue;
            logger.finest(featureFile.getAbsolutePath() + " doesn't seem to be an XML file.");
        }
    }

    protected void loadFeature(Uri parent, String xml) throws GadgetException {
        FeatureParser.ParsedFeature parsed = this.parser.parse(parent, xml);
        if (this.featureMap.containsKey(parsed.getName()) && logger.isLoggable(Level.WARNING)) {
            logger.warning("Overriding feature: " + parsed.getName() + " with def at: " + parent);
        }
        ArrayList bundles = Lists.newArrayList();
        for (FeatureParser.ParsedFeature.Bundle parsedBundle : parsed.getBundles()) {
            ArrayList resources = Lists.newArrayList();
            for (FeatureParser.ParsedFeature.Resource parsedResource : parsedBundle.getResources()) {
                if (parsedResource.getSource() == null) {
                    resources.add(new InlineFeatureResource(parsedResource.getContent()));
                    continue;
                }
                resources.add(this.resourceLoader.load(parsedResource.getSource(), parsedResource.getAttribs()));
            }
            bundles.add(new FeatureBundle(parsedBundle.getType(), parsedBundle.getAttribs(), resources));
        }
        this.featureMap.put(parsed.getName(), new FeatureNode(parsed.getName(), bundles, parsed.getDeps()));
    }

    private static class FeatureNode {
        private final String name;
        private final List<FeatureBundle> bundles;
        private final List<String> requestedDeps;
        private final List<FeatureNode> depList;
        private List<FeatureNode> transitiveDeps;
        private boolean calculatedDepsStale;
        private int nodeDepth = 0;

        private FeatureNode(String name, List<FeatureBundle> bundles, List<String> rawDeps) {
            this.name = name;
            this.bundles = Collections.unmodifiableList(bundles);
            this.requestedDeps = Collections.unmodifiableList(rawDeps);
            this.depList = Lists.newLinkedList();
            this.transitiveDeps = Lists.newArrayList((Object[])new FeatureNode[]{this});
            this.calculatedDepsStale = false;
        }

        public List<FeatureBundle> getBundles() {
            return this.bundles;
        }

        public List<String> getRawDeps() {
            return this.requestedDeps;
        }

        public void addDep(FeatureNode dep) {
            this.depList.add(dep);
            this.calculatedDepsStale = true;
        }

        private List<FeatureNode> getDepList() {
            ArrayList revOrderDeps = Lists.newArrayList(this.depList);
            Collections.reverse(revOrderDeps);
            return Collections.unmodifiableList(revOrderDeps);
        }

        public void completeNodeGraph() throws GadgetException {
            if (!this.calculatedDepsStale) {
                return;
            }
            this.nodeDepth = 0;
            this.transitiveDeps = Lists.newLinkedList();
            this.transitiveDeps.add(this);
            LinkedList toTraverse = Lists.newLinkedList();
            toTraverse.add(Pair.of((Object)this, (Object)Pair.of((Object)0, (Object)"")));
            while (!toTraverse.isEmpty()) {
                Pair next = (Pair)toTraverse.poll();
                String debug = (String)((Pair)next.two).two + ((Integer)((Pair)next.two).one > 0 ? " -> " : "") + ((FeatureNode)next.one).name;
                if (next.one == this && (Integer)((Pair)next.two).one != 0) {
                    throw new GadgetException(GadgetException.Code.INVALID_CONFIG, "Feature dep loop detected: " + debug);
                }
                this.transitiveDeps.add((FeatureNode)next.one);
                this.nodeDepth = Math.max(this.nodeDepth, (Integer)((Pair)next.two).one);
                for (FeatureNode nextDep : ((FeatureNode)next.one).getDepList()) {
                    toTraverse.add(Pair.of((Object)nextDep, (Object)Pair.of((Object)((Integer)((Pair)next.two).one + 1), (Object)debug)));
                }
            }
            Collections.reverse(this.transitiveDeps);
            this.calculatedDepsStale = false;
        }

        public List<FeatureNode> getTransitiveDeps() {
            return this.transitiveDeps;
        }
    }

    private static class FeatureBundle {
        private final String type;
        private final Map<String, String> attribs;
        private final List<FeatureResource> resources;

        private FeatureBundle(String type, Map<String, String> attribs, List<FeatureResource> resources) {
            this.type = type;
            this.attribs = Collections.unmodifiableMap(attribs);
            this.resources = Collections.unmodifiableList(resources);
        }

        public String getType() {
            return this.type;
        }

        public Map<String, String> getAttribs() {
            return this.attribs;
        }

        public List<FeatureResource> getResources() {
            return this.resources;
        }
    }

    private static class InlineFeatureResource
    extends FeatureResource.Default {
        private final String content;

        private InlineFeatureResource(String content) {
            this.content = content;
        }

        @Override
        public String getContent() {
            return this.content;
        }

        @Override
        public String getDebugContent() {
            return this.content;
        }
    }
}

